﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using WorldLab.GeoDataLib;
using WorldLab.MathLib;

namespace WorldLab.SampleAnalysisLib
{
    using SampleList = List<Sample>;

    /// analysis
    public class GridVarLayersAnalysis
    {
        
        private List<Grid> indepGridList;/// independent grid list
        private Grid depGrid;/// dependent grid
        private Grid weiGrid;/// weight grid
        private List<RectWndTemplate> tempList;/// template list
        private double tValue;
        private int gridCols;
        private int gridRows;
        private double noData;
        private RectWndTemplate template;

        public GridVarLayersAnalysis(List<Grid> indepGrdList, Grid depGrd, Grid weiGrd,
            List<RectWndTemplate> temList, double tVal)
        {
            indepGridList = indepGrdList;
            depGrid = depGrd;
            weiGrid = weiGrd;
            tempList = temList;
            tValue = tVal;
            gridRows = depGrd.RowCounts;
            gridCols = depGrd.ColCounts;
            noData = depGrd.NoDataValue;
            template = null;
        }

        private Range CalculateRangeOnTemplate(int c, int r)
        {
            int minX, maxX, minY, maxY;
            minX = c - template.CentreCol;
            maxX = c + (template.ColCounts - template.CentreCol - 1);
            minY = r - template.CentreRow;
            maxY = r + (template.RowCounts - template.CentreRow - 1);

            if (minX < 0)
            {
                minX = 0;
            }
            if (minY < 0)
            {
                minY = 0;
            }
            if (maxX > gridCols - 1)
            {
                maxX = gridCols - 1;
            }
            if (maxY > gridRows - 1)
            {
                maxY = gridRows - 1;
            }

            Range range = new Range(minX, maxX, minY, maxY);
            return range;
        }

        private SampleList GetSamplesOnTemplate(int c, int r)
        {
            Range range = CalculateRangeOnTemplate(c, r);
            SampleList result = new SampleList();
            Sample sample = null;
            int colOnTem, rowOnTem;
            TemplateCell cell;
            for (int i = range.MinX; i <= range.MaxX; ++i)
            {
                colOnTem = template.CentreCol + (i - c);
                for (int j = range.MinY; j < range.MaxY; ++j)
                {
                    rowOnTem = template.CentreRow + (j - r);
                    cell = template.GetElement(rowOnTem, colOnTem);
                    if (!cell.IsInWindow)
                    {
                        continue;
                    }
                    sample = new Sample(indepGridList.Count);
                    for (int k = 0; k < indepGridList.Count; ++k)
                    {
                        sample.IndepValues[k] = indepGridList[k].GetValue(j, i);
                    }
                    sample.DepValue = depGrid.GetValue(j, i);
                    sample.Weight = cell.Influence * weiGrid.GetValue(j, i);
                    result.Add(sample);
                }
            }
            return result;
        }

        private SampleList GetSamplesInStudyAreaOnTemplate(int c, int r)
        {
            Range range = CalculateRangeOnTemplate(c, r);
            SampleList result = new SampleList();
            Sample sample = null;
            int colOnTem, rowOnTem;
            TemplateCell cell;
            for (int i = range.MinX; i <= range.MaxX; ++i)
            {
                colOnTem = template.CentreCol + (i - c);
                for (int j = range.MinY; j < range.MaxY; ++j)
                {
                    if (depGrid.GetValue(j, i) == noData)
                    {
                        continue;
                    }
                    rowOnTem = template.CentreRow + (j - r);
                    cell = template.GetElement(rowOnTem, colOnTem);
                    if (!cell.IsInWindow)
                    {
                        continue;
                    }
                    sample = new Sample(indepGridList.Count);
                    for (int k = 0; k < indepGridList.Count; ++k)
                    {
                        sample.IndepValues[k] = indepGridList[k].GetValue(j, i);
                    }
                    sample.DepValue = depGrid.GetValue(j, i);
                    sample.Weight = cell.Influence * weiGrid.GetValue(j, i);
                    result.Add(sample);
                }
            }
            return result;
        }

        private SampleList GetSamplesNotNoDataOnTemplate(int c, int r)
        {
            Range range = CalculateRangeOnTemplate(c, r);
            SampleList result = new SampleList();
            Sample sample = null;
            int colOnTem, rowOnTem;
            TemplateCell cell;
            for (int i = range.MinX; i <= range.MaxX; ++i)
            {
                colOnTem = template.CentreCol + (i - c);
                for (int j = range.MinY; j < range.MaxY; ++j)
                {
                    if (depGrid.GetValue(j, i) == noData)
                    {
                        continue;
                    }
                    rowOnTem = template.CentreRow + (j - r);
                    cell = template.GetElement(rowOnTem, colOnTem);
                    if (!cell.IsInWindow)
                    {
                        continue;
                    }
                    sample = new Sample(indepGridList.Count);
                    for (int k = 0; k < indepGridList.Count; ++k)
                    {
                        if (indepGridList[k].GetValue(j, i) == noData)
                        {
                            sample.Dispose();
                            sample = null;
                            break;
                        }
                        sample.IndepValues[k] = indepGridList[k].GetValue(j, i);
                    }
                    if (sample != null)
                    {
                        sample.DepValue = depGrid.GetValue(j, i);
                        sample.Weight = cell.Influence * weiGrid.GetValue(j, i);
                        result.Add(sample);
                    }
                }
            }
            return result;
        }

        private SampleList GetSamplesOnConditionOne(int c, int r)
        {
            if (depGrid.GetValue(r, c) == noData)
            {
                return null;
            }

            for (int i = 0; i < indepGridList.Count; ++i)
            {
                if (indepGridList[i].GetValue(r, c) == noData)
                {
                    return null;
                }
            }

            SampleList samples = GetSamplesNotNoDataOnTemplate(r, c);
            Sample sample = null;
            if (samples.Count < indepGridList.Count)
            {
                while (samples.Count > 0)
                {
                    sample = samples[0];
                    samples.Remove(sample);
                    sample.Dispose();
                    sample = null;
                }
                samples = null;
            }

            return samples;
        }

        private SampleList GetSamplesOnConditionTwo(int c, int r,
            ref List<int> idxListOfNotVal, ref List<double> valueList, ref double t)
        {
            if (depGrid.GetValue(r, c) == noData)
            {
                return null;
            }

            List<SampleList> samplesList = new List<SampleList>();
            SampleList sampleList = null;
            List<SampleStatistic> sampleStatisticList = new List<SampleStatistic>();
            SampleStatistic sampleStatistic;
            for (int i = 0; i < tempList.Count; ++i)
            {
                template = tempList[i];
                sampleList = GetSamplesInStudyAreaOnTemplate(c, r);
                samplesList.Add(sampleList);
                sampleStatistic = new SampleStatistic(sampleList, indepGridList.Count);
                sampleStatisticList.Add(sampleStatistic);
            }

            int[] templeIndex = new int[indepGridList.Count];
            idxListOfNotVal = new List<int>();
            SampleStatistic.IndepStatistic indepStatistic = null;
            SampleStatistic.UnitStatistic unitStatisticOfMaxT = null;
            List<double> specialValueLsit = new List<double>();
            specialValueLsit.Add(noData);
            for (int i = 0; i < indepGridList.Count; ++i)
            {
                templeIndex[i] = -1;
                for (int j = 0; j < sampleStatisticList.Count; ++j)
                {
                    indepStatistic = sampleStatisticList[j].IndepStatis(i, specialValueLsit);
                    if (indepStatistic == null)
                    {
                        unitStatisticOfMaxT = null;
                    }
                    else
                    {
                        unitStatisticOfMaxT = indepStatistic.FindUnitStatisOfMaxTSta();
                        if (unitStatisticOfMaxT != null && unitStatisticOfMaxT.TStatistic >= tValue)
                        {
                            templeIndex[i] = j;
                            indepStatistic.Dispose();
                            indepStatistic = null;
                            break;
                        }
                        indepStatistic.Dispose();
                        indepStatistic = null;
                    }
                }
                if (templeIndex[i] == -1)
                {
                    idxListOfNotVal.Add(i);
                }
            }
 
            List<Sample> result = new List<Sample>();
            Sample sample = null;
            if (idxListOfNotVal.Count == indepGridList.Count)
            {
                for (int i = 0; i < samplesList[0].Count; ++i)
                {
                    sample = new Sample(samplesList[0][i]);
                    result.Add(sample);
                }
            }
            else
            {
                List<Sample> resultTmp = new List<Sample>();

                int maxIdx = templeIndex.Max();
                for (int i = 0; i < samplesList[maxIdx].Count; ++i)
                {
                    sample = new Sample(samplesList[maxIdx][i]);
                    resultTmp.Add(sample);
                }

                sampleStatistic = new SampleStatistic(samplesList[maxIdx], indepGridList.Count);
                valueList = new List<double>();
                t = 0.0;
                double value;
                for (int i = 0; i < indepGridList.Count; ++i)
                {
                    if (idxListOfNotVal.Contains(i))
                    {
                        continue;
                    }
                    indepStatistic = sampleStatistic.IndepStatis(i, specialValueLsit);
                    unitStatisticOfMaxT = indepStatistic.FindUnitStatisOfMaxTSta();
                    sampleList = sampleStatistic.IndepValueChange(i, unitStatisticOfMaxT.IndepValue,
                        unitStatisticOfMaxT.PossiveWeight, unitStatisticOfMaxT.NegativeWeight,
                        noData, 0.0);
                    for (int j = 0; j < resultTmp.Count; ++j)
                    {
                        resultTmp[j].IndepValues[i] = sampleList[j].IndepValues[i];
                    }
                    if (unitStatisticOfMaxT.TStatistic > t)
                    {
                        t = unitStatisticOfMaxT.TStatistic;
                    }
                    value = indepGridList[i].GetValue(r, c);
                    if (value == noData)
                    {
                        valueList.Add(0.0);
                    }
                    else
                    {
                        if (value <= unitStatisticOfMaxT.IndepValue)
                        {
                            valueList.Add(unitStatisticOfMaxT.PossiveWeight);
                        }
                        else
                        {
                            valueList.Add(unitStatisticOfMaxT.NegativeWeight);
                        }
                    }
                    indepStatistic.Dispose();
                    indepStatistic = null;
                    for (int j = 0; j < sampleList.Count; ++j)
                    {
                        sample = sampleList[j];
                        sample.Dispose();
                        sample = null;
                    }
                    sampleList.Clear();
                    sampleList = null;
                }
                sampleStatistic = null;

                double[] valValue = null;
                for (int i = 0; i < resultTmp.Count; ++i)
                {
                    int j = 0;
                    valValue = new double[indepGridList.Count - idxListOfNotVal.Count];
                    for (int k = 0; k < indepGridList.Count; ++k)
                    {
                        if (idxListOfNotVal.Contains(k))
                        {
                            continue;
                        }
                        valValue[j] = resultTmp[i].IndepValues[k];
                        j++;
                    }
                    sample = new Sample(j, valValue, resultTmp[i].DepValue, resultTmp[i].Weight);
                    result.Add(sample);
                }

                for (int i = 0; i < resultTmp.Count; ++i)
                {
                    sample = resultTmp[i];
                    sample.Dispose();
                    sample = null;
                }
                resultTmp.Clear();
                resultTmp = null;
            }

            specialValueLsit.Clear();
            templeIndex = null;

            for (int i = 0; i < sampleStatisticList.Count; ++i)
            {
                sampleStatistic = sampleStatisticList[i];
                sampleStatistic = null;
            }
            sampleStatisticList.Clear();

            for (int i = 0; i < samplesList.Count; ++i)
            {
                sampleList = samplesList[i];
                for (int j = 0; j < sampleList.Count; ++j)
                {
                    sample = sampleList[j];
                    sample.Dispose();
                    sample = null;
                }
                sampleList.Clear();
                sampleList = null;
            }
            samplesList.Clear();
            samplesList = null;

            return result;
        }

        public double ComputePI(int c, int r, 
            ref List<int> idxListOfNotVal, ref double[] beta, ref double t)
        {
            double pi = 0.0;
            List<double> valValues = null;
            SampleList samples = GetSamplesOnConditionTwo(c, r, 
                ref idxListOfNotVal, ref valValues, ref t);
            if (samples == null)
            {
                pi = noData;
            }
            else
            {
                int valIndepCount = indepGridList.Count - idxListOfNotVal.Count;
                SampleStatistic sampleStatistic = new SampleStatistic(samples, valIndepCount);
                if (idxListOfNotVal.Count == indepGridList.Count)
                {
                    pi = sampleStatistic.RatioOfDepEqu1Counts();
                }
                else
                {
                    int[] nums = sampleStatistic.GetDepEqu0And1Nums();
                    if (nums[0] == 0)
                    {
                        pi = 1.0;
                    }
                    else if (nums[1] == 0)
                    {
                        pi = 0.0;
                    }
                    else
                    {
                        SampleList samList = sampleStatistic.CombineSamples();
                        
                        Matrix xMatrix = new Matrix(samList.Count, valIndepCount + 1);
                        Vector yVector = new Vector(samList.Count);
                        Vector wVector = new Vector(samList.Count);
                        double[] xRow = null;
                        for (int i = 0; i < samList.Count; i++)
                        {
                            xRow = new double[valIndepCount + 1];
                            xRow[0] = 1.0;
                            for (int j = 0; j < valIndepCount; ++j)
                            {
                                xRow[j + 1] = samList[i].IndepValues[j];
                            }
                            xMatrix.SetRowVector(i, xRow);
                            yVector.SetElement(i, samList[i].DepValue);
                            wVector.SetElement(i, samList[i].Weight);
                        }
                        beta = LogisticRegression.ComputeBestBeta(xMatrix, yVector, wVector);
                        double gx = beta[0];
                        for (int i = 0; i < valIndepCount; ++i)
                        {
                            gx = gx + beta[i + 1] * valValues[i];
                        }
                        pi = 1.0 / (1.0 + Math.Exp(-gx));

                        xMatrix = null;
                        yVector = null;
                        wVector = null;
                        for (int i = 0; i < samList.Count; ++i)
                        {
                            samList[i] = null;
                        }
                        samList.Clear();
                        samList = null;
                    }
                    nums = null;
                }
                sampleStatistic = null;

                for (int i = 0; i < samples.Count; ++i)
                {
                    samples[i] = null;
                }
                samples.Clear();
                samples = null;
            }
            return pi;
        }

    }
}
